import itertools

try:
    import bpy
except:
    pass


def create(context, positions, height_factor):
    # order = compute_shortest_path(positions)
    # ordered_positions = [positions[i] for i in order]
    curve_obj = create_bounce_curve(positions, height_factor)
    context.report({'INFO'}, "Bounce curve created.")
    return curve_obj


def compute_shortest_path(positions):
    n = len(positions)
    if n <= 10:
        return compute_shortest_path_brute_force(positions)
    else:
        return compute_nearest_neighbor_path(positions)


def compute_shortest_path_brute_force(positions):
    n = len(positions)
    indices = range(n)
    min_total_distance = None
    best_path = None
    for perm in itertools.permutations(indices):
        total_distance = sum(
            (positions[perm[i + 1]] - positions[perm[i]]).length for i in range(n - 1)
        )
        if min_total_distance is None or total_distance < min_total_distance:
            min_total_distance = total_distance
            best_path = perm
    return best_path


def compute_nearest_neighbor_path(positions):
    n = len(positions)
    min_total_distance = None
    best_path = None
    for start in range(n):
        indices = set(range(n))
        path = [start]
        indices.remove(start)
        current = start
        total_distance = 0
        while indices:
            current_pos = positions[current]
            next_index = min(
                indices, key=lambda i: (positions[i] - current_pos).length
            )
            min_dist = (positions[next_index] - current_pos).length
            total_distance += min_dist
            indices.remove(next_index)
            path.append(next_index)
            current = next_index
        if min_total_distance is None or total_distance < min_total_distance:
            min_total_distance = total_distance
            best_path = path
    return best_path


def create_bounce_curve(ordered_positions, height_factor):
    points = []
    handle_types = []

    for i in range(len(ordered_positions) - 1):
        # Original point
        p0 = ordered_positions[i]
        points.append(p0)
        handle_types.append('FREE')  # Use 'FREE' to manually set handle positions

        # Midpoint
        p1 = ordered_positions[i + 1]
        mid_point = (p0 + p1) / 2

        # Calculate distance between p0 and p1
        distance = (p1 - p0).length

        # Adjust height based on distance
        mid_point.z += height_factor * distance

        points.append(mid_point)
        handle_types.append('AUTO')  # Midpoints have 'AUTO' handles

    # Add the last original point
    points.append(ordered_positions[-1])
    handle_types.append('FREE')  # Use 'FREE' for the last point

    # Reverse the points and handle_types to reverse the curve direction
    points.reverse()
    handle_types.reverse()

    # Create a new curve
    curve_data = bpy.data.curves.new(name='SimpleBounceCurve', type='CURVE')
    curve_data.dimensions = '3D'
    spline = curve_data.splines.new('BEZIER')
    spline.bezier_points.add(len(points) - 1)

    # Assign points and handle types
    for i, (point, handle_type) in enumerate(zip(points, handle_types)):
        bp = spline.bezier_points[i]
        bp.co = point
        bp.handle_left_type = handle_type
        bp.handle_right_type = handle_type

        if handle_type == 'FREE':
            # Set handle positions to the control point position (zero-length handles)
            bp.handle_left = bp.co.copy()
            bp.handle_right = bp.co.copy()

    # Create a new object with the curve
    curve_obj = bpy.data.objects.new('bounce_curve', curve_data)
    # Add the object to the scene
    bpy.context.collection.objects.link(curve_obj)
    return curve_obj
