import random
import time

try:
    import bpy
    from . import scene
    from .. import utils
except ModuleNotFoundError as e:
    from . import scene
    from .. import utils

collection_layout = 'mc_layout_assist'
collection_animation = 'mc_layout_animation'


def check_rhythm_times_str(rhythm_times_str):
    if rhythm_times_str is None or len(rhythm_times_str) == 0:
        return False, "请设置节奏点"
    try:
        rhy_points = rhythm_times_str.split(",")
        for p in rhy_points:
            float(p)
    except:
        return False, "节奏点格式错误"
    if len(rhy_points) < 1:
        return False, "请至少设置一个节奏点"
    return True, None


def check_params(ball):
    if ball is None:
        return False, "请选择小球对象"
    return True, None


def fill_expiration(scene, expired_at):
    scene.mc_expire_tips = expired_at


class UpdateAnimationOperator(bpy.types.Operator):
    bl_idname = "object.update_animation"
    bl_label = "更新小球动画"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        scene = context.scene
        ball = scene.mc_ball
        camera = scene.mc_camera
        background = scene.mc_background
        bpm = scene.mc_bpm
        pad = scene.mc_pad
        key = scene.mc_auth_key.strip()
        secret = scene.mc_auth_secret.strip()
        # track_support = scene.mc_track_support
        support = scene.mc_support
        speed_factor = scene.mc_speed_factor
        jump_factor = scene.mc_jump_factor
        rhythm_points_str = scene.mc_rhythm_points
        update_layout = scene.mc_update_layout
        update_camera_animation = scene.mc_update_camera_animation
        layout_seed = scene.mc_layout_seed
        material_track = scene.mc_track_material
        material_track_support = scene.mc_track_support_material
        ###### camera
        camera_remain_distance = scene.mc_camera_remain_distance
        camera_offset = (0, -scene.mc_camera_offset_y, scene.mc_camera_offset_z)
        camera_rotation = (scene.mc_camera_rotation_x, 0, 0)
        ######
        if not update_camera_animation:
            camera = None

        luminous = (scene.mc_pad_pre_luminous, scene.mc_pad_post_luminous)
        result, msg = check_params(ball)
        frame_rate = bpy.context.scene.render.fps

        if not result:
            self.report({'ERROR'}, msg)
            return {'CANCELLED'}

        ball_radius = utils.get_ball_radius(ball)
        if ball_radius is None or ball_radius <= 0:
            self.report({'ERROR'}, "小球对象必须网格化")
            return {'CANCELLED'}

        if scene.mc_beat_source == '音轨标记':
            rhythm_frames = [m.frame for m in bpy.context.scene.timeline_markers]
            rhythm_times = [f / frame_rate for f in rhythm_frames]
        else:
            result, msg = check_rhythm_times_str(rhythm_points_str)
            if not result:
                self.report({'ERROR'}, msg)
                return {'CANCELLED'}
            rhythm_times = [float(p) for p in rhythm_points_str.split(",")]
        if len(rhythm_times) < 1:
            self.report({'ERROR'}, "请至少设置一个节奏点")
            return {'CANCELLED'}
        # remove duplicated number
        rhythm_times = list(set(rhythm_times))
        rhythm_times.sort()
        rhythm_times.extend([rhythm_times[-1] + 60 / bpm / 2, rhythm_times[-1] + 60 / bpm * 4])

        bpy.context.scene.frame_end = int(rhythm_times[-1] * frame_rate)

        if update_layout:
            b, msg, layout, meta = utils.get_layout(key, secret, bpm, speed_factor, frame_rate, rhythm_times,
                                                    ball_radius, layout_seed)
            if not b:
                self.report({'ERROR'}, msg)
                return {'CANCELLED'}
            utils.remove_collection(collection_layout)
            for i, p in enumerate(layout):
                utils.create_ball('mc_layout_ball_' + str(i + 1), p, collection_layout, ball_radius)
        camera_info = {'remain_distance': camera_remain_distance, 'offset': camera_offset,
                       'rotation': camera_rotation}
        objs_info = utils.get_objects_info({'pad': pad, 'support': support})
        objs_info.update({'camera': camera_info})
        positions = [p for p in utils.get_ball_positions(collection_layout)]
        if len(positions) != len(rhythm_times):
            self.report({'ERROR'}, "鼓点数量变化，需要勾选更新小球关键位置")
            return {'CANCELLED'}
        b, msg, data, meta = utils.get_animation(key, secret, -9.8 * 10 * jump_factor, frame_rate, rhythm_times,
                                                 positions,
                                                 ball_radius, luminous,
                                                 objs_info)
        if not b:
            self.report({'ERROR'}, msg)
            return {'CANCELLED'}
        fill_expiration(context.scene, meta.get('expired_at', ''))
        utils.set_animation(data, collection_animation, ball, pad, support, background, camera, material_track,
                            material_track_support)

        return {'FINISHED'}


class RandomLayoutSeedOperator(bpy.types.Operator):
    bl_idname = "object.rand_layout_seed"
    bl_label = "随机编号"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        random.seed(time.time() * 1000000)
        context.scene.mc_layout_seed = random.randint(0, 999999)
        return {'FINISHED'}


class MusicBallPanel(bpy.types.Panel):
    bl_label = "音乐小球"
    # bl_label = "Music Ball"
    bl_idname = "TOOL_PT_mercury_client_panel"
    bl_space_type = 'PROPERTIES'
    bl_region_type = 'WINDOW'
    # bl_category = 'ABC'
    bl_context = 'world'

    @classmethod
    def poll(cls, context):
        return True

    def draw(self, context):
        layout = self.layout
        layout.use_property_split = True
        # layout.use_property_decorate = False
        layout.label(text="全局配置")
        layout.row().prop(context.scene, "mc_expire_tips", placeholder='无需填写')
        layout.row().prop(context.scene, "mc_auth_key")
        layout.row().prop(context.scene, "mc_auth_secret")

        layout.separator()

        layout.label(text="场景配置")
        layout.row().prop(context.scene, "mc_ball")
        layout.row().prop(context.scene, "mc_camera")
        camera_adv_h, camera_adv_b = layout.panel('panel_camera_adv', default_closed=True)
        camera_adv_h.label(text='相机高级设置')
        if camera_adv_b:
            camera_adv_b.row().prop(context.scene, "mc_camera_remain_distance")
            camera_adv_b.row().prop(context.scene, "mc_camera_offset_y")
            camera_adv_b.row().prop(context.scene, "mc_camera_offset_z")
            camera_adv_b.row().prop(context.scene, "mc_camera_rotation_x")

        layout.row().prop(context.scene, "mc_background")
        layout.row().prop(context.scene, "mc_pad")
        layout.row().prop(context.scene, "mc_support")
        layout.row().prop(context.scene, "mc_track_material")
        layout.row().prop(context.scene, "mc_track_support_material")
        layout.separator()

        layout.row().prop(context.scene, "mc_beat_source")
        if context.scene.mc_beat_source == '自定义':
            layout.row().prop(context.scene, "mc_rhythm_points")
        layout.row().prop(context.scene, "mc_bpm")
        layout.row().prop(context.scene, "mc_jump_factor")
        layout.row().prop(context.scene, "mc_speed_factor")
        layout.row().prop(context.scene, "mc_pad_pre_luminous")
        layout.row().prop(context.scene, "mc_pad_post_luminous")
        layout.separator()
        layout.row().prop(context.scene, "mc_update_layout")
        if context.scene.mc_update_layout:
            row = layout.row()
            row.prop(context.scene, "mc_layout_seed")
            row.operator(RandomLayoutSeedOperator.bl_idname)

        layout.row().prop(context.scene, "mc_update_camera_animation")
        layout.operator(UpdateAnimationOperator.bl_idname)
        layout.separator()
